package net.senohrabek.marcus.client

import io.ktor.client.*
import io.ktor.client.call.*
import io.ktor.client.plugins.contentnegotiation.*
import io.ktor.client.request.*
import io.ktor.client.statement.*
import io.ktor.http.*
import io.ktor.serialization.kotlinx.json.*
import io.ktor.utils.io.jvm.javaio.*
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.SerializationStrategy
import kotlinx.serialization.Serializer
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.decodeFromStream
import net.senohrabek.marcus.client.exceptions.ClientErrorException
import net.senohrabek.marcus.client.exceptions.ServerErrorException
import net.senohrabek.marcus.search.*
import java.io.Closeable

class MarcusClient(
    private val endpoint: String,
    private val client: HttpClient
) : Closeable {
    suspend fun getFields(index: String, request: GetFieldsRequest): GetFieldsResponse {
        val response = client.get {
            buildUrl(endpoint, "index", index, "document", "fields")
            contentType(ContentType.Application.Json)
            setBody(request)
        }
        return response.getResponseDataOrThrow()
    }

    suspend fun search(index: String, searchRequest: SearchRequest): SearchResponse {
        val response = client.get {
            buildUrl(endpoint, "index", index, "document")
            contentType(ContentType.Application.Json)
            setBody(searchRequest)
        }
        return response.getResponseDataOrThrow()
    }

    suspend fun explain(index: String, explainRequest: ExplainRequest): ExplainResponse {
        val response = client.get {
            buildUrl(endpoint, "index", index, "document")
            contentType(ContentType.Application.Json)
            setBody(explainRequest)
        }
        return response.getResponseDataOrThrow()
    }

    @OptIn(ExperimentalSerializationApi::class)
    private suspend inline fun <reified TResponse> HttpResponse.getResponseDataOrThrow(): TResponse {
        if (status.value in 400..499) {
            throw ClientErrorException(status, bodyAsText())
        } else if (status.value >= 500) {
            throw ServerErrorException(status, bodyAsText())
        } else return body()
    }

    companion object {
        val defaultClient = HttpClient {
            install(ContentNegotiation) {
                json()
            }
        }
    }

    override fun close() {
        client.close()
    }
}